(() => {
  // =========================================================
  // OLD BROWSER GUARD (include this in all future web apps)
  // =========================================================
  function browserIsTooOld(){
    try{
      return !(
        'Promise' in window &&
        'Map' in window &&
        'Set' in window &&
        'fetch' in window &&
        'localStorage' in window &&
        'FileReader' in window &&
        'Blob' in window &&
        'URL' in window &&
        'addEventListener' in window &&
        window.CSS && CSS.supports && CSS.supports('color', 'var(--x)')
      );
    }catch(e){
      return true;
    }
  }

  const el = {
    sheet: document.getElementById('sheet'),
    cellsLayer: document.getElementById('cellsLayer'),
    colHeaders: document.getElementById('colHeaders'),
    rowHeaders: document.getElementById('rowHeaders'),
    editor: document.getElementById('editor'),
    editorTa: document.getElementById('editorTa'),
    nameBox: document.getElementById('nameBox'),
    formulaBar: document.getElementById('formulaBar'),
    tabs: document.getElementById('tabs'),
    statusText: document.getElementById('statusText'),
    roBadge: document.getElementById('roBadge'),

    btnOpen: document.getElementById('btnOpen'),
    fileOpen: document.getElementById('fileOpen'),
    btnSaveXlsx: document.getElementById('btnSaveXlsx'),
    btnSaveCsv: document.getElementById('btnSaveCsv'),
    btnAddSheet: document.getElementById('btnAddSheet'),

    modal: document.getElementById('modal'),
    modalTitle: document.getElementById('modalTitle'),
    modalMsg: document.getElementById('modalMsg'),
    modalClose: document.getElementById('modalClose'),
  };

  function showModal(title, html){
    el.modalTitle.textContent = title;
    el.modalMsg.innerHTML = html;
    el.modal.style.display = 'flex';
  }
  el.modalClose.onclick = () => el.modal.style.display = 'none';
  el.modal.onclick = (e) => { if(e.target === el.modal) el.modal.style.display = 'none'; };

  if(browserIsTooOld()){
    showModal(
      'Browser not supported',
      `Your browser is not compatible with this web app<br><br>
       Please download the latest browser: Google Chrome, Microsoft Edge, or Mozilla Firefox<br><br>
       After updating, reopen this page`
    );
    document.querySelectorAll('button, input, select, textarea').forEach(x => x.disabled = true);
    return;
  }

  if(typeof XLSX === 'undefined'){
    showModal(
      'Excel library not loaded',
      `File import/export needs <b>./js/xlsx.full.min.js</b><br><br>
       Please confirm the file exists and the path is correct`
    );
  }

  // ---------- Grid config ----------
  const ROWS = 1000;
  const COLS = 200;
  const rowH = 24;
  const colW = 110;

  // ---------- Helpers ----------
  const clamp = (v, a, b) => Math.max(a, Math.min(b, v));
  const keyOf = (r,c) => `${r},${c}`;
  const isFormula = (s) => typeof s === 'string' && s.trim().startsWith('=');

  function colToName(n){
    let s = '';
    n = n + 1;
    while(n > 0){
      const m = (n - 1) % 26;
      s = String.fromCharCode(65 + m) + s;
      n = Math.floor((n - 1) / 26);
    }
    return s;
  }
  function nameToCol(name){
    let n = 0;
    for (const ch of name.toUpperCase()){
      if (ch < 'A' || ch > 'Z') break;
      n = n * 26 + (ch.charCodeAt(0) - 64);
    }
    return n - 1;
  }
  function parseA1(ref){
    const m = /^([A-Z]+)(\d+)$/.exec(ref.toUpperCase().trim());
    if(!m) return null;
    const c = nameToCol(m[1]);
    const r = parseInt(m[2], 10) - 1;
    if(Number.isNaN(r) || c < 0) return null;
    return {r, c};
  }
  function a1(r,c){ return `${colToName(c)}${r+1}`; }

  function escapeHtml(s){
    return String(s)
      .replaceAll('&','&amp;')
      .replaceAll('<','&lt;')
      .replaceAll('>','&gt;')
      .replaceAll('"','&quot;')
      .replaceAll("'","&#039;");
  }

  // ---------- Workbook model ----------
  function blankSheet(name){
    return { name, readOnly:false, cells:new Map() }; // key "r,c" => { raw }
  }
  const workbook = { sheets:[blankSheet('Sheet1')], active:0 };
  const activeSheet = () => workbook.sheets[workbook.active];

  function workbookReadOnly(){ return !!activeSheet().readOnly; }
  function setReadOnly(flag){
    activeSheet().readOnly = !!flag;
    el.roBadge.style.display = workbookReadOnly() ? 'inline-block' : 'none';
    el.btnAddSheet.disabled = workbookReadOnly();
  }

  // ---------- Persistence ----------
  const LS_KEY = 'freehtmlapp_excelclone_v1';
  function serialize(){
    return {
      active: workbook.active,
      sheets: workbook.sheets.map(s => ({
        name: s.name,
        readOnly: !!s.readOnly,
        cells: [...s.cells.entries()]
      }))
    };
  }
  function applySerialized(obj){
    workbook.sheets = (obj.sheets || []).map(s => {
      const sh = blankSheet(s.name || 'Sheet');
      sh.readOnly = !!s.readOnly;
      for(const [k,v] of (s.cells || [])){
        if(v) sh.cells.set(k, v);
      }
      return sh;
    });
    if(workbook.sheets.length === 0) workbook.sheets = [blankSheet('Sheet1')];
    workbook.active = clamp(obj.active ?? 0, 0, workbook.sheets.length - 1);
    setReadOnly(!!activeSheet().readOnly);
  }
  function saveLocal(){
    try{
      localStorage.setItem(LS_KEY, JSON.stringify(serialize()));
      setStatus('Saved');
    }catch(e){
      setStatus('Save failed');
    }
  }
  function loadLocal(){
    try{
      const s = localStorage.getItem(LS_KEY);
      if(!s) return false;
      applySerialized(JSON.parse(s));
      setStatus('Loaded');
      return true;
    }catch(e){ return false; }
  }

  // ---------- Status ----------
  function setStatus(msg){
    el.statusText.textContent = msg;
    clearTimeout(setStatus._t);
    setStatus._t = setTimeout(() => el.statusText.textContent = 'Ready', 900);
  }

  // ---------- Selection ----------
  const selection = { r:0, c:0, rect:{r1:0,c1:0,r2:0,c2:0}, dragging:false, anchor:null };

  function inRect(r,c, rect){
    const r1 = Math.min(rect.r1, rect.r2);
    const r2 = Math.max(rect.r1, rect.r2);
    const c1 = Math.min(rect.c1, rect.c2);
    const c2 = Math.max(rect.c1, rect.c2);
    return r>=r1 && r<=r2 && c>=c1 && c<=c2;
  }

  // ---------- Virtual rendering ----------
  const rendered = new Map();
  let viewport = { r1:0,r2:0,c1:0,c2:0, scTop:0, scLeft:0 };

  function measureViewport(){
    const scTop = el.sheet.scrollTop;
    const scLeft = el.sheet.scrollLeft;
    const w = el.sheet.clientWidth;
    const h = el.sheet.clientHeight;
    viewport = {
      r1: clamp(Math.floor(scTop / rowH) - 2, 0, ROWS-1),
      r2: clamp(Math.ceil((scTop + h) / rowH) + 2, 0, ROWS-1),
      c1: clamp(Math.floor(scLeft / colW) - 2, 0, COLS-1),
      c2: clamp(Math.ceil((scLeft + w) / colW) + 2, 0, COLS-1),
      scTop, scLeft
    };
  }

  function renderHeaders(){
    // Column headers
    el.colHeaders.innerHTML = '';
    const colLayer = document.createElement('div');
    colLayer.style.position = 'relative';
    colLayer.style.height = '28px';
    colLayer.style.width = (COLS * colW) + 'px';
    colLayer.style.transform = `translateX(${-viewport.scLeft}px)`;
    el.colHeaders.appendChild(colLayer);

    for(let c=viewport.c1;c<=viewport.c2;c++){
      const d = document.createElement('div');
      d.className = 'col-header-cell' + (c === selection.c ? ' active' : '');
      d.textContent = colToName(c);
      d.style.left = (c * colW) + 'px';
      d.style.width = colW + 'px';
      colLayer.appendChild(d);
    }

    // Row headers
    el.rowHeaders.innerHTML = '';
    const rowLayer = document.createElement('div');
    rowLayer.style.position = 'relative';
    rowLayer.style.width = '56px';
    rowLayer.style.height = (ROWS * rowH) + 'px';
    rowLayer.style.transform = `translateY(${-viewport.scTop}px)`;
    el.rowHeaders.appendChild(rowLayer);

    for(let r=viewport.r1;r<=viewport.r2;r++){
      const d = document.createElement('div');
      d.className = 'row-header-cell' + (r === selection.r ? ' active' : '');
      d.textContent = String(r+1);
      d.style.top = (r * rowH) + 'px';
      d.style.height = rowH + 'px';
      rowLayer.appendChild(d);
    }
  }

  function getRaw(r,c){
    return activeSheet().cells.get(keyOf(r,c))?.raw ?? '';
  }

  function renderCells(){
    const sh = activeSheet();
    const keep = new Set();

    for(let r=viewport.r1;r<=viewport.r2;r++){
      for(let c=viewport.c1;c<=viewport.c2;c++){
        const k = keyOf(r,c);
        keep.add(k);

        let div = rendered.get(k);
        if(!div){
          div = document.createElement('div');
          div.className = 'cell';
          div.dataset.r = r;
          div.dataset.c = c;
          el.cellsLayer.appendChild(div);
          rendered.set(k, div);
        }

        div.style.left = (c * colW) + 'px';
        div.style.top  = (r * rowH) + 'px';

        const raw = sh.cells.get(k)?.raw ?? '';
        div.textContent = raw;
        div.title = raw ? `${a1(r,c)}: ${raw}` : a1(r,c);

        let cls = 'cell';
        if(selection.r === r && selection.c === c) cls += ' selected';
        if(selection.rect && inRect(r,c, selection.rect)) cls += ' in-range';
        div.className = cls;
      }
    }

    for(const [k,div] of rendered.entries()){
      if(!keep.has(k)){
        div.remove();
        rendered.delete(k);
      }
    }
  }

  function renderTabs(){
    el.tabs.innerHTML = '';
    workbook.sheets.forEach((sh, i) => {
      const t = document.createElement('div');
      t.className = 'tab' + (i === workbook.active ? ' active' : '');
      t.textContent = sh.name + (sh.readOnly ? ' (RO)' : '');
      t.addEventListener('click', () => {
        workbook.active = i;
        selection.r=0; selection.c=0; selection.rect={r1:0,c1:0,r2:0,c2:0};
        setReadOnly(!!activeSheet().readOnly);
        renderAll();
        saveLocal();
      });
      t.addEventListener('dblclick', () => {
        const name = prompt('Rename sheet', sh.name);
        if(!name) return;
        sh.name = name.trim().slice(0,31) || sh.name;
        renderTabs();
        saveLocal();
      });
      el.tabs.appendChild(t);
    });
  }

  function updateFormulaUI(){
    el.nameBox.value = a1(selection.r, selection.c);
    el.formulaBar.value = getRaw(selection.r, selection.c);
  }

  function renderAll(){
    measureViewport();
    renderHeaders();
    renderCells();
    renderTabs();
    updateFormulaUI();
  }

  // ---------- Editor ----------
  function ensureVisible(r,c){
    const top = r * rowH, left = c * colW;
    const bottom = top + rowH, right = left + colW;
    const vTop = el.sheet.scrollTop, vLeft = el.sheet.scrollLeft;
    const vBottom = vTop + el.sheet.clientHeight;
    const vRight  = vLeft + el.sheet.clientWidth;

    if(top < vTop) el.sheet.scrollTop = top;
    else if(bottom > vBottom) el.sheet.scrollTop = bottom - el.sheet.clientHeight;

    if(left < vLeft) el.sheet.scrollLeft = left;
    else if(right > vRight) el.sheet.scrollLeft = right - el.sheet.clientWidth;
  }

  function setSelection(r,c, extend=false){
    r = clamp(r, 0, ROWS-1);
    c = clamp(c, 0, COLS-1);
    if(!extend){
      selection.r=r; selection.c=c;
      selection.rect = {r1:r,c1:c,r2:r,c2:c};
    }else{
      selection.rect.r2=r; selection.rect.c2=c;
      selection.r=r; selection.c=c;
    }
    ensureVisible(r,c);
    renderAll();
  }

  function openEditor(){
    if(workbookReadOnly()){
      setStatus('Read only');
      return;
    }
    const r = selection.r, c = selection.c;
    const raw = getRaw(r,c);

    const left = c * colW - el.sheet.scrollLeft;
    const top  = r * rowH - el.sheet.scrollTop;

    el.editor.style.left = (left + 2) + 'px';
    el.editor.style.top = (top + 2) + 'px';
    el.editor.style.width = (colW - 4) + 'px';
    el.editor.style.height = (rowH * 2) + 'px';
    el.editor.style.display = 'block';

    el.editorTa.value = raw;
    el.editorTa.focus();
    el.editorTa.select();
  }

  function closeEditor(commit){
    if(el.editor.style.display !== 'block') return;

    if(commit && !workbookReadOnly()){
      const r = selection.r, c = selection.c;
      const k = keyOf(r,c);
      const v = el.editorTa.value ?? '';
      if(v.trim() === '') activeSheet().cells.delete(k);
      else activeSheet().cells.set(k, { raw: v });
      saveLocal();
      setStatus('Updated ' + a1(r,c));
    }

    el.editor.style.display = 'none';
    el.editorTa.value = '';
    renderAll();
  }

  // ---------- Pointer-based selection (mouse + touch) ----------
  el.cellsLayer.addEventListener('pointerdown', (e) => {
    const t = e.target.closest('.cell');
    if(!t) return;
    el.cellsLayer.setPointerCapture(e.pointerId);

    const r = +t.dataset.r, c = +t.dataset.c;
    selection.dragging = true;
    selection.anchor = {r,c};
    selection.r=r; selection.c=c;
    selection.rect = {r1:r,c1:c,r2:r,c2:c};
    renderAll();
  });

  el.cellsLayer.addEventListener('pointermove', (e) => {
    if(!selection.dragging) return;
    const rect = el.sheet.getBoundingClientRect();
    const x = e.clientX - rect.left + el.sheet.scrollLeft;
    const y = e.clientY - rect.top + el.sheet.scrollTop;
    const c = clamp(Math.floor(x / colW), 0, COLS-1);
    const r = clamp(Math.floor(y / rowH), 0, ROWS-1);
    selection.rect = {r1:selection.anchor.r, c1:selection.anchor.c, r2:r, c2:c};
    selection.r=r; selection.c=c;
    renderAll();
  });

  el.cellsLayer.addEventListener('pointerup', () => {
    selection.dragging = false;
    selection.anchor = null;
  });

  el.cellsLayer.addEventListener('dblclick', (e) => {
    const t = e.target.closest('.cell');
    if(!t) return;
    setSelection(+t.dataset.r, +t.dataset.c, false);
    openEditor();
  });

  el.sheet.addEventListener('scroll', () => {
    if(el.editor.style.display === 'block') closeEditor(true);
    requestAnimationFrame(renderAll);
  });

  // Click outside editor commits
  window.addEventListener('pointerdown', (e) => {
    if(el.editor.style.display !== 'block') return;
    if(e.target === el.editor || el.editor.contains(e.target)) return;
    closeEditor(true);
  });

  // ---------- Keyboard ----------
  window.addEventListener('keydown', (e) => {
    if(el.editor.style.display === 'block'){
      if(e.key === 'Enter' && !e.shiftKey){
        e.preventDefault();
        closeEditor(true);
        setSelection(selection.r+1, selection.c, false);
      }else if(e.key === 'Escape'){
        e.preventDefault();
        closeEditor(false);
      }
      return;
    }

    if((e.ctrlKey || e.metaKey) && e.key.toLowerCase() === 's'){
      e.preventDefault();
      saveAsXlsx();
      return;
    }

    const extend = e.shiftKey;
    if(e.key === 'ArrowUp'){ e.preventDefault(); setSelection(selection.r-1, selection.c, extend); return; }
    if(e.key === 'ArrowDown'){ e.preventDefault(); setSelection(selection.r+1, selection.c, extend); return; }
    if(e.key === 'ArrowLeft'){ e.preventDefault(); setSelection(selection.r, selection.c-1, extend); return; }
    if(e.key === 'ArrowRight'){ e.preventDefault(); setSelection(selection.r, selection.c+1, extend); return; }

    if(e.key === 'Enter' || e.key === 'F2'){ e.preventDefault(); openEditor(); return; }

    // Typing starts edit
    if(e.key.length === 1 && !e.ctrlKey && !e.metaKey && !e.altKey){
      if(workbookReadOnly()){ setStatus('Read only'); return; }
      openEditor();
      el.editorTa.value = e.key;
      el.editorTa.setSelectionRange(1,1);
      e.preventDefault();
    }
  });

  // ---------- Copy/Paste TSV ----------
  window.addEventListener('copy', (e) => {
    if(el.editor.style.display === 'block') return;
    const rect = selection.rect;
    const r1 = Math.min(rect.r1, rect.r2), r2 = Math.max(rect.r1, rect.r2);
    const c1 = Math.min(rect.c1, rect.c2), c2 = Math.max(rect.c1, rect.c2);

    const rows = [];
    for(let r=r1;r<=r2;r++){
      const cols = [];
      for(let c=c1;c<=c2;c++) cols.push(getRaw(r,c));
      rows.push(cols.join('\t'));
    }
    e.clipboardData.setData('text/plain', rows.join('\n'));
    e.preventDefault();
    setStatus('Copied');
  });

  window.addEventListener('paste', (e) => {
    if(el.editor.style.display === 'block') return;
    if(workbookReadOnly()){ setStatus('Read only'); return; }

    const txt = e.clipboardData.getData('text/plain');
    if(!txt) return;
    e.preventDefault();

    const startR = selection.r, startC = selection.c;
    const grid = txt.replace(/\r/g,'').split('\n').map(r => r.split('\t'));

    for(let i=0;i<grid.length;i++){
      for(let j=0;j<grid[i].length;j++){
        const r = startR + i, c = startC + j;
        if(r>=ROWS || c>=COLS) continue;
        const raw = grid[i][j] ?? '';
        const k = keyOf(r,c);
        if(raw.trim()==='') activeSheet().cells.delete(k);
        else activeSheet().cells.set(k, { raw });
      }
    }
    saveLocal();
    renderAll();
    setStatus('Pasted');
  });

  // ---------- Name box / formula bar ----------
  el.nameBox.addEventListener('keydown', (e) => {
    if(e.key !== 'Enter') return;
    const p = parseA1(el.nameBox.value);
    if(!p){ setStatus('Invalid cell'); return; }
    setSelection(p.r, p.c, false);
  });

  el.formulaBar.addEventListener('keydown', (e) => {
    if(e.key !== 'Enter') return;
    if(workbookReadOnly()){ setStatus('Read only'); return; }
    const v = el.formulaBar.value ?? '';
    const k = keyOf(selection.r, selection.c);
    if(v.trim()==='') activeSheet().cells.delete(k);
    else activeSheet().cells.set(k, { raw: v });
    saveLocal();
    setSelection(selection.r+1, selection.c, false);
  });

  // ---------- Sheets ----------
  el.btnAddSheet.onclick = () => {
    if(workbookReadOnly()){ setStatus('Read only'); return; }
    workbook.sheets.push(blankSheet('Sheet' + (workbook.sheets.length + 1)));
    workbook.active = workbook.sheets.length - 1;
    selection.r=0; selection.c=0; selection.rect={r1:0,c1:0,r2:0,c2:0};
    setReadOnly(false);
    renderAll();
    saveLocal();
  };

  // ---------- XLSX Import / Export ----------
  function safeFileBaseName(){
    const d = new Date();
    const pad = (n) => String(n).padStart(2,'0');
    return `freehtml.app_spreadsheet_${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())}_${pad(d.getHours())}-${pad(d.getMinutes())}`;
  }

  function downloadArrayBuffer(buf, filename, mime){
    const blob = new Blob([buf], {type: mime || 'application/octet-stream'});
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();
    URL.revokeObjectURL(url);
  }

  function buildSheetJSWorkbook(){
    if(typeof XLSX === 'undefined') throw new Error('XLSX library not loaded');

    const wb = XLSX.utils.book_new();

    workbook.sheets.forEach((sh, si) => {
      const ws = {};
      let maxR = 0, maxC = 0;

      for(const k of sh.cells.keys()){
        const [r,c] = k.split(',').map(Number);
        if(r > maxR) maxR = r;
        if(c > maxC) maxC = c;
      }
      maxR = Math.min(maxR, ROWS-1);
      maxC = Math.min(maxC, COLS-1);

      for(const [k, obj] of sh.cells.entries()){
        const [r,c] = k.split(',').map(Number);
        if(r < 0 || c < 0 || r > maxR || c > maxC) continue;

        const addr = XLSX.utils.encode_cell({r,c});
        const raw = obj?.raw ?? '';

        if(isFormula(raw)){
          ws[addr] = { t:'s', f: raw.trim().slice(1), v: '' };
        }else{
          ws[addr] = { t:'s', v: String(raw) };
        }
      }

      ws['!ref'] = XLSX.utils.encode_range({s:{r:0,c:0}, e:{r:maxR,c:maxC}});
      XLSX.utils.book_append_sheet(wb, ws, (sh.name || `Sheet${si+1}`).slice(0,31));
    });

    return wb;
  }

  function saveAsXlsx(){
    try{
      const wb = buildSheetJSWorkbook();
      const out = XLSX.write(wb, {bookType:'xlsx', type:'array', compression:true});
      downloadArrayBuffer(
        out,
        `${safeFileBaseName()}.xlsx`,
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
      );
      setStatus('Saved XLSX');
    }catch(err){
      showModal('Save failed', `Unable to save XLSX<br><br>Please ensure <b>js/xlsx.full.min.js</b> is present`);
    }
  }

  function saveActiveAsCsv(){
    try{
      const wb = buildSheetJSWorkbook();
      const name = wb.SheetNames[workbook.active];
      const ws = wb.Sheets[name];
      const csv = XLSX.utils.sheet_to_csv(ws);

      const blob = new Blob([csv], {type:'text/csv;charset=utf-8'});
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${safeFileBaseName()}_${name}.csv`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      URL.revokeObjectURL(url);
      setStatus('Saved CSV');
    }catch(err){
      showModal('CSV export failed', 'Unable to export CSV for this sheet');
    }
  }

  async function openWorkbookFromFile(file){
    if(typeof XLSX === 'undefined'){
      showModal('Open failed', `Excel import needs <b>js/xlsx.full.min.js</b>`);
      return;
    }

    try{
      const ext = (file.name.split('.').pop() || '').toLowerCase();
      const buf = await file.arrayBuffer();

      const wb = XLSX.read(buf, { type:'array', cellFormula:true, cellDates:true });

      const limitations = [];
      let readOnly = false;

      if(ext === 'xlsm'){
        limitations.push('This file is XLSM (macro-enabled). Macros cannot run in this web app');
        readOnly = true;
      }

      const newState = { active:0, sheets:[] };
      for(const name of wb.SheetNames){
        const ws = wb.Sheets[name];
        const sh = blankSheet(name);

        if(ws && ws['!merges'] && ws['!merges'].length){
          limitations.push(`Merged cells detected in "${name}" (editing simplified)`);
          readOnly = true;
        }

        const ref = ws && ws['!ref'] ? ws['!ref'] : 'A1:A1';
        const range = XLSX.utils.decode_range(ref);

        const maxR = clamp(range.e.r, 0, ROWS-1);
        const maxC = clamp(range.e.c, 0, COLS-1);

        for(let r=range.s.r; r<=maxR; r++){
          for(let c=range.s.c; c<=maxC; c++){
            const addr = XLSX.utils.encode_cell({r,c});
            const cell = ws[addr];
            if(!cell) continue;

            let raw = '';
            if(cell.f) raw = '=' + String(cell.f);
            else raw = String(cell.v ?? '');

            sh.cells.set(keyOf(r,c), { raw });
          }
        }

        sh.readOnly = readOnly;
        newState.sheets.push(sh);
      }

      applySerialized(newState);
      setReadOnly(readOnly);
      renderAll();
      saveLocal();

      if(readOnly){
        showModal(
          'Opened with limitations (Read Only)',
          `This file may not be editable 100% in a web app<br><br>
           We are showing as much data as possible in <b>read-only mode</b><br><br>
           For full editing and exact Excel behaviour, use <b>Microsoft Excel</b><br><br>
           <b>Detected limitations</b><br>
           <ul style="margin:8px 0 0 18px">${limitations.map(x => `<li>${escapeHtml(x)}</li>`).join('')}</ul>`
        );
      }else{
        setStatus('Opened ' + file.name);
      }
    }catch(err){
      showModal('Open failed', 'Unable to read this file in the browser. Please try another file or use MS Excel');
    }
  }

  el.btnOpen.onclick = () => {
    el.fileOpen.value = '';
    el.fileOpen.click();
  };

  el.fileOpen.onchange = () => {
    const f = el.fileOpen.files?.[0];
    if(!f) return;
    openWorkbookFromFile(f);
  };

  el.btnSaveXlsx.onclick = saveAsXlsx;
  el.btnSaveCsv.onclick = saveActiveAsCsv;

  // ---------- Init layer sizes ----------
  el.cellsLayer.style.width = (COLS * colW) + 'px';
  el.cellsLayer.style.height = (ROWS * rowH) + 'px';

  // ---------- Boot ----------
  loadLocal();
  setReadOnly(!!activeSheet().readOnly);
  renderAll();

  setInterval(saveLocal, 15000);
})();
